# Utility routines
function Reset_Folder {
	param(
		[string] $Path
	)

    # Take ownership of the folder
    takeown.exe /f "$Path" /r /a /d Y | Out-Null
    if ($LASTEXITCODE -ne 0) {
		Write-Host "Non-zero return code from TakeOwn, rc = $LASTEXITCODE"
    }

    # Reset permissions
    $escapedPath = $Path.Replace("\","\\")
    $oFile = Get-WmiObject -class CIM_Directory -Filter "Name='$escapedPath'"	
    $oSD = ([wmiclass]'Win32_SecurityDescriptor').psbase.CreateInstance()
    try {
        $iRc = $oFile.ChangeSecurityPermissionsEx($oSD, 4, $null, $true)
    } 
	catch
	{
		Write-Host "Unable to change security permissions: $_"
        $iRc = 9999
    }
    return $iRc
}

function Reset_File {
	param(
		[string] $Path
	)

    # Take ownership of the file
    takeown.exe /f "$Path" /a | Out-Null
    if ($LASTEXITCODE -ne 0) {
		Write-Host "Non-zero return code from TakeOwn, rc = $LASTEXITCODE"
    }

    # Reset permissions
    $escapedPath = $Path.Replace("\","\\")
    $oFile = Get-WmiObject -class CIM_DataFile -Filter "Name='$escapedPath'"
    $oSD = ([wmiclass]'Win32_SecurityDescriptor').psbase.CreateInstance()
    try {
        $iRc = $oFile.ChangeSecurityPermissions($oSD, 4)
    } catch {
		Write-Host "Unable to change security permissions: $_"
        $iRc = 9998
    }
    return $iRc
}

function Clean_Path {
	param(
		[string] $Path
	)

    # Remove folders
	Get-ChildItem -Path $Path -Directory | % {
		$folder = $_.Name
		$folderPath = $_.FullName
        switch ($folder) {
            "_t" { 
				Write-Host "Skipping folder $folder"
			}
            "recycler" { 
				Write-Host "Skipping folder $folder" 
			}
            "system volume information" {
				Write-Host "Skipping folder $folder" 
			}
            default {
                # Try to remove the folder
				Write-Host "Removing folder $folder" 
                cmd.exe /c rd /s /q "$folderPath" | Out-Null
                # If it still exists, reset the permissions and try again
                if (Test-Path $folderPath) {
                    # Reset the permissions
                    $iRc = Reset_Folder($folderPath)
                    if ($iRc -ne 0) {
                        Write-Host "Non-zero return code resetting folder $folderPath, RC = $iRc"
                    }
                    # Try the delete again
					cmd.exe /c rd /s /q "$folderPath" | Out-Null
                    if ($LASTEXITCODE -ne 0) {
                        Write-Host "Unable to delete folder $folderPath, RC = $iRc"
                    }
                }	
        	}
    	}
	}

	# Remove files at the root of the path
	Get-ChildItem -Path $Path -File | % {
		$file = $_.FullName
        Reset_File $file | Out-Null
		Write-Host "Removing file: $file"
		Remove-Item $file -Force
	}
}
# Find the ADK
try
{
    $kitsRoot = Get-ItemPropertyValue -Path "HKLM:\Software\Microsoft\Windows Kits\Installed Roots" -Name KitsRoot10
}
catch
{
    Write-Error "ADK is not installed."
    return 1
}

# Find Windows PE
$peRoot = $kitsRoot + "Assessment and Deployment Kit\Windows Preinstallation Environment\"
if (-not (Test-Path $peRoot)) {
    Write-Error "Windows PE is not installed."
    return 2
}

$platforms = @("amd64", "x86", "arm64")

$platforms | % {

    $platform = $_

    # Copy the winpe.wim
    $peFile = "$peRoot\$platform\en-us\winpe.wim";
    if (-not (Test-Path $peFile)) {
        Write-Error "Windows PE file " + $peFile + " does not exist."
        return 3
    }
    $peNew = [Environment]::GetFolderPath("MyDocuments") + "\winpe_$platform.wim"
    Copy-Item -Path $peFile -Destination $peNew -Force

    # Mount the winpe.wim
    $peMount = "$($env:TEMP)\mount_$platform"
    if (-not (Test-Path $peMount)) {
        MkDir $peMount
    }
    Mount-WindowsImage -Path $peMount -ImagePath $peNew -Index 1

    # Add the needed components to it
    $packagePath = "$peRoot\$platform\WinPE_OCs"
    ("winpe-scripting", "winpe-wmi", "winpe-securestartup", "winpe-netfx", "winpe-powershell", "winpe-storagewmi") | % {
        Write-Host "Adding package $_"
        Add-WindowsPackage -Path $peMount -PackagePath "$packagePath\$($_).cab"
        Add-WindowsPackage -Path $peMount -PackagePath "$packagePath\en-us\$($_)_en-us.cab"    
    }

    # Inject any needed drivers
    if (Test-Path ".\$platform\Drivers") {
        Write-Host "Injecting drivers from .\$platform\Drivers"
        Add-WindowsDriver -Path $peMount -Driver ".\$platform\Drivers" -Recurse
    }

    # Inject any needed update
    if (Test-Path ".\$platform\Updates") {
        Write-Host "Injecting updates from .\$platform\Updates"
        Dir ".\$platform\Updates" | % { Add-WindowsPackage -Path $peMount -PackagePath ".\$platform\Updates\$_" }
    }

    # Remove the old background
    takeown /f "$peMount\Windows\system32\winpe.jpg"
    icacls "$peMount\Windows\system32\winpe.jpg" /grant everyone:f
    Remove-Item "$peMount\Windows\system32\winpe.jpg"

    # Add the extra files
    Copy-Item -Path "winpeshl.ini" -Destination "$peMount\windows\system32" -Force
    Copy-Item -Path "Unattend_PE_$platform.xml" -Destination "$peMount\unattend.xml" -Force
    Copy-Item -Path "bootstrap.vbs" -Destination "$peMount\bootstrap.vbs" -Force
    Copy-Item -Path "winpe.jpg" -Destination "$peMount\Windows\system32\winpe.jpg" -Force
    # Copy-Item -Path "startnet.cmd" -Destination "$peMount\Windows\system32\startnet.cmd" -Force

    # Clean up uneeded garbage
    if ($platform -eq 'amd64') {
        @("$peMount\windows\SysWOW64", "$peMount\windows\Microsoft.NET\Framework") | % { Clean_Path $_}
        @("$peMount\windows\WinSXS\wow64_*", "$peMount\windows\WinSXS\x86_*", "$peMount\windows\Microsoft.NET\assembly\GAC_32\*") | % {
            Get-Item $_ | % { Clean_Path $_ }
        }
    }

    # Unmount and commit
    Dismount-WindowsImage -Path $peMount -Save

    # Export the image to reduce the size
    $peClean = [Environment]::GetFolderPath("MyDocuments") + "\Clean_winpe_$platform.wim"
    Export-WindowsImage -SourceImagePath $peNew -DestinationImagePath $peClean -SourceIndex 1 -Setbootable

    # Report completion
    Write-Host "Windows PE generated: $peNew"
}
